import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.io.IOException;

/**
 * Diese Klasse stellt einige statische Methoden zum Einlesen von primitiven
 * Datentypen und Zeichenketten von der Standardeingabe zur Verfügung.
 *
 * @author pscb
 * @version 0.2
 */
public class StdIn {

	private static final BufferedReader br = new BufferedReader(
			new InputStreamReader(System.in));

	/**
	 * Präsentiert dem Benutzer auf der Standardausgabe eine Eingabeaufforderung
	 * und liest eine durch einen Zeilenumbruch abgeschlossene Zeichenkette von
	 * der Standardeingabe. Der Zeilenumbruch gehört nicht zur zurückgegebenen
	 * Zeichenkette.
	 * 
	 * @param prompt	Eingabeaufforderung
	 * @return			eingegebene Zeichenkette
	 */
	public static String askString(String prompt) {
		System.out.print(prompt);
		try {
			return br.readLine();
		} catch (IOException e) {
			System.err.println("could not read from stdin");
			System.exit(1);
			return null;
		}
	}

	/**
	 * Präsentiert dem Benutzer auf der Standardausgabe eine Eingabeaufforderung
	 * und liest eine ganze Zahl von der Standardeingabe. Wenn keine ganze Zahl
	 * eingegeben wurde, wird mit einer Fehlermeldung erneut zur Eingabe
	 * aufgefordert.
	 * 
	 * @param prompt	Eingabeaufforderung
	 * @return			eingegebene ganze Zahl
	 */
	public static int askInt(String prompt) {
		do {
			try {
				return Integer.parseInt(askString(prompt));
			} catch (NumberFormatException e) {
				System.err.println("not an integer value");
			}
		} while (true);
	}

	/**
	 * Präsentiert dem Benutzer auf der Standardausgabe eine Eingabeaufforderung
	 * und liest eine Fließkomma-Zahl von der Standardeingabe. Wenn keine
	 * Fließkomma-Zahl eingegeben wurde, wird mit einer Fehlermeldung erneut zur
	 * Eingabe aufgefordert.
	 * 
	 * @param prompt	Eingabeaufforderung
	 * @return			eigegebene Fließkommazahl
	 */
	public static double askDouble(String prompt) {
		do {
			try {
				return Double.parseDouble(askString(prompt));
			} catch (NumberFormatException e) {
				System.err.println("not a double value");
			}
		} while (true);
	}

	/**
	 * Präsentiert dem Benutzer auf der Standardausgabe eine Eingabeaufforderung
	 * und liest eine durch einen Zeilenumbruch abgeschlossene Zeichenkette von der
	 * Standardeingabe. Wenn die Zeichenkette einer der beiden vorgegebenen
	 * Alternativen entspricht, wird der zugehörige Wahrheitswert zurück gegeben und
	 * ansonsten mit einer Fehlermeldung erneut zur Eingabe aufgefordert.
	 * 
	 * @param prompt	Eingabeaufforderung
	 * @param altTrue	erwartete Eingabe für Rückgabe von <code>true</code>
	 * @param altFalse	erwartete Eingabe für Rückgabe von <code>false</code>
	 * @return			Wahrheitswert entsprechend der eingegebenen Zeichenkette
	 */
	public static boolean askBoolean(String prompt, String altTrue,
			String altFalse) {
		do {
			String s = askString(prompt);
			if (s.equals(altTrue))
				return true;
			else if (s.equals(altFalse))
				return false;
			else
				System.out.println("enter either \"" + altTrue + "\" or \""
						+ altFalse + "\"");
		} while (true);
	}

	/**
	 * Hauptprogramm zum Test der zuvor definierten Methoden
	 * 
	 * @param args	Kommandozeilenargumente werden ignoriert
	 */
	public static void main(String[] args) {
		String s = StdIn.askString("Bitte eine Zeichenkette eingeben: ");
		System.out.println(s);

		int i = StdIn.askInt("Bitte eine ganze Zahl eingeben: ");
		System.out.println(i);

		double d = StdIn.askDouble("Bitte eine Zahl eingeben: ");
		System.out.println(d);

		boolean b = StdIn.askBoolean("Bitte \"ja\" oder \"nein\" eingeben: ",
				"ja", "nein");
		System.out.println(b);
	}

}