import java.awt.Button;
import java.awt.Dialog;
import java.awt.Frame;
import java.awt.GridLayout;
import java.awt.Label;
import java.awt.Panel;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.List;

/**
 * Diese Klasse erzeugt die Dialoge zur bungsteilnehmerverwaltung. Dazu gehren
 * die Anzeige der bungsgruppen, Auswahl der Aktionen Anzeigen, Einschreiben,
 * Verschieben und Lschen von Teilnehmern.
 * 
 * @author mradicke
 */
public class UebungsverwaltungGUI extends Panel {

	/**
	 * Das bergeordnete Fenster in dem diese Klasse grafisch eingebettet wird.
	 */
	private Frame frame;
	/**
	 * Die bungsverwaltung.
	 */
	private Uebungsverwaltung uebungsverwaltung;
	/**
	 * Der ursprngliche Fenstertitel zur Anzeige des Benutzers und der
	 * gewhlten Gruppe.
	 */
	private String titel;

	/**
	 * Der Standard-Konstruktor erzeugt eine neue bungsverwaltung und zeigt die
	 * bungsgruppenauswahl an.
	 * 
	 * @param frame
	 *            bergeordnetes Fenster
	 */
	public UebungsverwaltungGUI(Frame frame) {
		this.frame = frame;
		titel = frame.getTitle();
		uebungsverwaltung = new Uebungsverwaltung();
		zeigeAuswahlPanel();
	}

	/**
	 * Eine bersicht der bungsgruppen wird angezeigt. Eine Gruppe kann
	 * ausgewhlt werden. Nach Auswahl einer Gruppe werden mgliche Aktionen fr
	 * diese Gruppe angezeigt.
	 */
	private void zeigeAuswahlPanel() {
		removeAll();
		frame.setTitle(titel);
		add(new Label("Vorlesung"));
		add(new Label("Tag"));
		add(new Label("Uhrzeit"));
		add(new Label("Raum"));
		add(new Label("Tutor"));
		add(new Label("Studiengang"));
		add(new Label("Kapazitaet"));
		add(new Label("Teilnehmer"));
		add(new Label("Aktion"));

		List<Uebungsgruppe> uebungsgruppenListe = uebungsverwaltung
				.holeUebungsgruppenListe();
		int anzahl = uebungsgruppenListe.size();
		for (final Uebungsgruppe gruppe : uebungsgruppenListe) {
			add(new Label(gruppe.holeVorlesung()));
			add(new Label(gruppe.holeTag()));
			add(new Label(gruppe.holeUhrzeit()));
			add(new Label(gruppe.holeRaum()));
			add(new Label(gruppe.holeTutor().holeName()));
			add(new Label(gruppe.holeStudiengangListe().toString()));
			add(new Label("" + gruppe.holeKapazitaet()));
			add(new Label("" + gruppe.holeTeilnehmerListe().size()));
			Button auswahlButton = new Button("Auswahl");
			auswahlButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					uebungsverwaltung.auswaehlen(gruppe);
					Uebungsgruppe gruppe = uebungsverwaltung
							.holeAusgewaehlteUebungsgruppe();
					frame.setTitle(titel + " - gewaehlte Gruppe: "
							+ gruppe.holeVorlesung() + ", " + gruppe.holeTag()
							+ ", " + gruppe.holeUhrzeit());
					zeigeAktionsPanel();
				}
			});
			add(auswahlButton);
		}
		setLayout(new GridLayout(anzahl + 1, 9));
		validate();
	}

	/**
	 * Es werden mgliche Aktionen fr die ausgewhlte Gruppe angezeigt.
	 * Aktionen sind Teilnehmer anzeigen, verschieben, lschen, einschreiben.
	 */
	private void zeigeAktionsPanel() {
		removeAll();
		Button anzeigenButton = new Button("Teilnehmer anzeigen");
		anzeigenButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				zeigeTeilnehmerAnzeigePanel();
			}
		});
		Button verschiebenButton = new Button("Teilnehmer verschieben");
		verschiebenButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				zeigeTeilnehmerVerschiebenPanel();
			}
		});
		Button loeschenButton = new Button("Teilnehmer loeschen");
		loeschenButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				zeigeTeilnehmerLoeschenPanel();
			}
		});
		Button eintragenButton = new Button("In Gruppe eintragen");
		eintragenButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				try {
					uebungsverwaltung.eintragen((Student) Nutzerverwaltung
							.holeAngemeldeterNutzer());
					zeigeMeldung("Anmeldung erfolgreich!");
				} catch (Exception f) {
					zeigeMeldung(f.getMessage());
				}
				zeigeAuswahlPanel();
			}
		});

		add(anzeigenButton);
		if (Nutzerverwaltung.holeAngemeldeterNutzer() instanceof Tutor) {
			add(verschiebenButton);
			add(loeschenButton);
		}
		if (Nutzerverwaltung.holeAngemeldeterNutzer() instanceof Student) {
			add(eintragenButton);
		}
		setLayout(new GridLayout(5, 1));
		Button zurueckButton = new Button("Zurueck");
		zurueckButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				zeigeAuswahlPanel();
			}
		});
		add(zurueckButton);
		validate();
	}

	/**
	 * Die Teilnehmer der gewhlten bungsgruppe werden angezeigt.
	 */
	private void zeigeTeilnehmerAnzeigePanel() {
		removeAll();
		add(new Label("Name"));
		add(new Label("Vorname"));
		add(new Label("Matrikelnummer"));
		add(new Label("Studiengang"));

		List<Student> teilnehmerListe = uebungsverwaltung
				.holeAusgewaehlteUebungsgruppe().holeTeilnehmerListe();
		int anzahl = teilnehmerListe.size();
		for (final Student teilnehmer : teilnehmerListe) {
			add(new Label(teilnehmer.holeName()));
			add(new Label(teilnehmer.holeVorname()));
			add(new Label("" + teilnehmer.holeMatrikelnummer()));
			add(new Label(teilnehmer.holeStudiengang()));
		}
		setLayout(new GridLayout(anzahl + 2, 4));
		Button zurueckButton = new Button("Zurueck");
		zurueckButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				zeigeAktionsPanel();
			}
		});
		add(zurueckButton);
		for (int i = 1; i < 4; i++) {
			add(new Panel());
		}
		validate();
	}

	/**
	 * Ein Teilnehmer der gewhlten bungsgruppe kann verschoben werden. Dazu
	 * werden alle Teilnehmer der gewhlten bungsgruppe angezeigt.
	 */
	private void zeigeTeilnehmerVerschiebenPanel() {
		removeAll();
		add(new Label("Name"));
		add(new Label("Vorname"));
		add(new Label("Matrikelnummer"));
		add(new Label("Studiengang"));
		add(new Label("Aktion"));

		List<Student> teilnehmerListe = uebungsverwaltung
				.holeAusgewaehlteUebungsgruppe().holeTeilnehmerListe();
		int anzahl = teilnehmerListe.size();
		for (final Student teilnehmer : teilnehmerListe) {
			add(new Label(teilnehmer.holeName()));
			add(new Label(teilnehmer.holeVorname()));
			add(new Label("" + teilnehmer.holeMatrikelnummer()));
			add(new Label(teilnehmer.holeStudiengang()));
			Button verschiebenButton = new Button("Verschieben");
			verschiebenButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					zeigeVerschiebenZielePanel(teilnehmer);
				}
			});
			add(verschiebenButton);
		}
		setLayout(new GridLayout(anzahl + 2, 5));
		Button zurueckButton = new Button("Zurueck");
		zurueckButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				zeigeAktionsPanel();
			}
		});
		add(zurueckButton);
		for (int i = 1; i < 5; i++) {
			add(new Panel());
		}
		validate();
	}

	/**
	 * bungsgruppen werden angezeigt, um als Ziel einer Teilnehmerverschiebung
	 * ausgewhlt zu werden.
	 * 
	 * @param teilnehmer
	 *            zu verschiebender Teilnehmer
	 */
	private void zeigeVerschiebenZielePanel(final Student teilnehmer) {
		removeAll();
		add(new Label("Vorlesung"));
		add(new Label("Tag"));
		add(new Label("Uhrzeit"));
		add(new Label("Raum"));
		add(new Label("Tutor"));
		add(new Label("Studiengang"));
		add(new Label("Kapazitaet"));
		add(new Label("Teilnehmer"));
		add(new Label("Aktion"));

		List<Uebungsgruppe> uebungsgruppenListe = uebungsverwaltung
				.holeUebungsgruppenListe();
		int anzahl = uebungsgruppenListe.size();
		for (final Uebungsgruppe gruppe : uebungsgruppenListe) {
			add(new Label(gruppe.holeVorlesung()));
			add(new Label(gruppe.holeTag()));
			add(new Label(gruppe.holeUhrzeit()));
			add(new Label(gruppe.holeRaum()));
			add(new Label(gruppe.holeTutor().holeName()));
			add(new Label(gruppe.holeStudiengangListe().toString()));
			add(new Label("" + gruppe.holeKapazitaet()));
			add(new Label("" + gruppe.holeTeilnehmerListe().size()));
			Button zielButton = new Button("VerschiebeZiel");
			zielButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					try {
						uebungsverwaltung.verschiebe(teilnehmer, gruppe);
						zeigeMeldung("Verschiebung erfolgreich!");
					} catch (Exception f) {
						zeigeMeldung(f.getMessage());
					}
					zeigeAuswahlPanel();
				}
			});
			add(zielButton);
		}
		setLayout(new GridLayout(anzahl + 2, 9));
		Button zurueckButton = new Button("Zurueck");
		zurueckButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				zeigeTeilnehmerVerschiebenPanel();
			}
		});
		add(zurueckButton);
		for (int i = 1; i < 9; i++) {
			add(new Panel());
		}
		validate();
	}

	/**
	 * Ein Teilnehmer der ausgewhlten bungsgruppe kann gelscht werden. Dazu
	 * werden alle Teilnehmer der gewhlten bungsgruppe angezeigt.
	 */
	private void zeigeTeilnehmerLoeschenPanel() {
		removeAll();
		add(new Label("Name"));
		add(new Label("Vorname"));
		add(new Label("Matrikelnummer"));
		add(new Label("Studiengang"));
		add(new Label("Aktion"));

		List<Student> teilnehmerListe = uebungsverwaltung
				.holeAusgewaehlteUebungsgruppe().holeTeilnehmerListe();
		int anzahl = teilnehmerListe.size();
		for (final Student teilnehmer : teilnehmerListe) {
			add(new Label(teilnehmer.holeName()));
			add(new Label(teilnehmer.holeVorname()));
			add(new Label("" + teilnehmer.holeMatrikelnummer()));
			add(new Label(teilnehmer.holeStudiengang()));
			Button loeschenButton = new Button("Loeschen");
			loeschenButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					uebungsverwaltung.loesche(teilnehmer);
					zeigeMeldung("Loeschen erfolgreich!");
					zeigeAuswahlPanel();
				}
			});
			add(loeschenButton);
		}
		setLayout(new GridLayout(anzahl + 2, 5));
		Button zurueckButton = new Button("Zurueck");
		zurueckButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				zeigeAktionsPanel();
			}
		});
		add(zurueckButton);
		for (int i = 1; i < 5; i++) {
			add(new Panel());
		}
		validate();
	}

	/**
	 * Zeigt eine Erfolgs- bzw. Fehlermeldung.
	 * 
	 * @param meldung
	 *            Nachricht der Meldung.
	 */
	private void zeigeMeldung(String meldung) {
		final Dialog meldungDialog = new Dialog(frame);
		meldungDialog.setLayout(new GridLayout(2, 1));
		meldungDialog.add(new Label(meldung));
		meldungDialog.setTitle(meldung);
		meldungDialog.setModal(true);
		Button ok = new Button("OK");
		ok.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				meldungDialog.dispose();
			}
		});
		meldungDialog.add(ok);
		meldungDialog.setSize(300, 100);
		meldungDialog.setVisible(true);
		meldungDialog.validate();
	}

}
