/*
	Author:				Torben Brodt
	Matrikelnummer:		451314
	Institut:			FH Wiesbaden
	Datum:				14. Juni 2007
*/

// Headerdateien einbinden
#include "aquarium.h"

/**
 * Liefert den Normalenvektor aus einer Flaeche von 3 Punkten
 * @param *res -> double[3] Array mit x,y,z Koordinate
 * @param x1, y1, z1 -> 3D Punkt Nr. 1
 * @param x2, y2, z2 -> 3D Punkt Nr. 2
 * @param x3, y3, z3 -> 3D Punkt Nr. 3
 */
void normalenVektor(float *res, float x1, float y1, float z1, float x2, float y2, float z2, float x3, float y3, float z3) 
{
	float laenge;
	float v1x, v1y, v1z;
	float v2x, v2y, v2z;

	// Als Parameter erhalten wir 3 Punkte
	// Daraus machen wir 2 Vektoren
	v1x = x1-x3;
	v1y = y1-y3;
	v1z = z1-z3;

	v2x = x2-x3;
	v2y = y2-y3;
	v2z = z2-z3;

	// Als Parameter erhalten wir 2 Vektoren
	// daraus errechnen wir das Kreuzprodukt
	res[0] = (v1y * v2z) - (v1z * v2y);  
	res[1] = (v1z * v2x) - (v1x * v2z);
	res[2] = (v1x * v2y) - (v1y * v2x);

	// Laenge ermitteln
	laenge = sqrt(pow(res[0], 2) + pow(res[1], 2) + pow(res[2], 2));

	// Auf Einheitsvektor bringen, indem wir durch Laenge teilen
	res[0] /= laenge;
	res[1] /= laenge;
	res[2] /= laenge;
	
	return; 
}

/**
 * orthographische Projektion fuers Cockpit
 * Quellen: http://www.lighthouse3d.com/opengl/glut/index.php?bmpfontortho
 */
void setOrthographicProjection(void) 
{
	// switch to projection mode
	glMatrixMode(GL_PROJECTION);
	// save previous matrix which contains the 
	//settings for the perspective projection
	glPushMatrix();
	// reset matrix
	glLoadIdentity();
	// set a 2D orthographic projection
	gluOrtho2D(0, w, 0, h);
	// invert the y axis, down is positive
	glScalef(1, -1, 1);
	// mover the origin from the bottom left corner
	// to the upper left corner
	glTranslatef(0, -h, 0);
	glMatrixMode(GL_MODELVIEW);
}

/**
 * Reset-Methode fuer die orthographische Projektion
 */
void resetPerspectiveProjection(void) 
{
	// set the current matrix to GL_PROJECTION
	glMatrixMode(GL_PROJECTION);
	// restore previous settings
	glPopMatrix();
	// get back to GL_MODELVIEW matrix
	glMatrixMode(GL_MODELVIEW);
}

/**
 * Zeichnet das Cockpit mit seiner Framerate
 * Quellen: http://www.lighthouse3d.com/opengl/glut/index.php?fps
 */
void drawCockpit(void) 
{
	char *c;

	// FPS
	frame++;
	fpstime=glutGet(GLUT_ELAPSED_TIME);
	if (fpstime - timebase > 500) {
		sprintf(string, "FPS: %4.2f\n", frame*1000.0/(fpstime-timebase));
		timebase = fpstime;
		frame = 0;
	}

	glColor3f(0.0f,1.0f,1.0f);
	setOrthographicProjection();
	glPushMatrix();
	glLoadIdentity();
		// set position to start drawing fonts
		glRasterPos2f(0.6, 0.9);
		// loop all the characters in the string
		for (c=string; *c != '\0'; c++) {
			glutBitmapCharacter((void*)GLUT_BITMAP_8_BY_13, *c);
		}
	glPopMatrix();
	resetPerspectiveProjection();
}

/**
 * Zeichnet die Pflanze im Aquarium
 * Quellen: http://www.codeworx.org/opengl_tut6.php
 */
void drawPflanze(void) 
{
	// Texturierung
	glBindTexture(GL_TEXTURE_2D , pflanze);
	glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_EYE_LINEAR);
	glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, GL_EYE_LINEAR);

	glEnable(GL_TEXTURE_2D);
	
	glPushMatrix();
		glTranslatef(13, -14, 24);
		glColor4f(1, 1, 1, 0.5); // weisser BG 
		
		glBegin(GL_QUADS);
			// Das vordere QUAD
			glTexCoord2f(0.0f, 0.0f); glVertex3f(-1.0f, -1.0f, 1.0f); 
			// unten links an der Form und der Textur
			glTexCoord2f(1.0f, 0.0f); glVertex3f( 1.0f, -1.0f, 1.0f); 
			// unten rechts an der Form und der Textur
			glTexCoord2f(1.0f, 1.0f); glVertex3f( 1.0f, 1.0f, 1.0f); 
			// oben rechts an der Form und der Textur
			glTexCoord2f(0.0f, 1.0f); glVertex3f(-1.0f, 1.0f, 1.0f); 
			// oben links an der Form und der Textur

			// Das hintere QUAD
			glTexCoord2f(1.0f, 0.0f); glVertex3f(-1.0f, -1.0f, -1.0f);
			// unten rechts an der Form und der Textur
			glTexCoord2f(1.0f, 1.0f); glVertex3f(-1.0f, 1.0f, -1.0f);
			// oben rechts an der Form und der Textur
			glTexCoord2f(0.0f, 1.0f); glVertex3f( 1.0f, 1.0f, -1.0f);
			// oben links an der Form und der Textur
			glTexCoord2f(0.0f, 0.0f); glVertex3f( 1.0f, -1.0f, -1.0f);
			// unten links an der Form und der Textur

			// Das obere QUAD
			glTexCoord2f(0.0f, 1.0f); glVertex3f(-1.0f, 1.0f, -1.0f); 
			// oben links an der Form und der Textur
			glTexCoord2f(0.0f, 0.0f); glVertex3f(-1.0f, 1.0f, 1.0f); 
			// unten links an der Form und der Textur
			glTexCoord2f(1.0f, 0.0f); glVertex3f( 1.0f, 1.0f, 1.0f); 
			// unten rechts an der Form und der Textur
			glTexCoord2f(1.0f, 1.0f); glVertex3f( 1.0f, 1.0f, -1.0f); 
			// oben rechts an der Form und der Textur

			// Das untere QUAD
			glTexCoord2f(1.0f, 1.0f); glVertex3f(-1.0f, -1.0f, -1.0f); 
			// oben rechts an der Form und der Textur
			glTexCoord2f(0.0f, 1.0f); glVertex3f( 1.0f, -1.0f, -1.0f);
			// oben links an der Form und der Textur
			glTexCoord2f(0.0f, 0.0f); glVertex3f( 1.0f, -1.0f, 1.0f); 
			// unten links an der Form und der Textur
			glTexCoord2f(1.0f, 0.0f); glVertex3f(-1.0f, -1.0f, 1.0f); 
			// unten rechts an der Form und der Textur

			// Das rechte QUAD
			glTexCoord2f(1.0f, 0.0f); glVertex3f( 1.0f, -1.0f, -1.0f); 
			// unten rechts an der Form und der Textur
			glTexCoord2f(1.0f, 1.0f); glVertex3f( 1.0f, 1.0f, -1.0f); 
			// oben rechts an der Form und der Textur
			glTexCoord2f(0.0f, 1.0f); glVertex3f( 1.0f, 1.0f, 1.0f); 
			// oben links an der Form und der Textur
			glTexCoord2f(0.0f, 0.0f); glVertex3f( 1.0f, -1.0f, 1.0f); 
			// unten links an der Form und der Textur

			// Das linke QUAD
			glTexCoord2f(0.0f, 0.0f); glVertex3f(-1.0f, -1.0f, -1.0f); 
			// unten links an der Form und der Textur
			glTexCoord2f(1.0f, 0.0f); glVertex3f(-1.0f, -1.0f, 1.0f); 
			// unten rechts an der Form und der Textur
			glTexCoord2f(1.0f, 1.0f); glVertex3f(-1.0f, 1.0f, 1.0f); 
			// oben rechts an der Form und der Textur
			glTexCoord2f(0.0f, 1.0f); glVertex3f(-1.0f, 1.0f, -1.0f); 
			// oben links an der Form und der Textur
		glEnd();
	glPopMatrix();

	glDisable(GL_TEXTURE_GEN_S);
	glDisable(GL_TEXTURE_GEN_T);
	glDisable(GL_TEXTURE_2D);
}

/**
 * Zeichnet die Umgebung
 */
void drawAthmosphere(void) 
{
	glDisable(GL_COLOR_MATERIAL);

	// Licht nach oben
	GLfloat position2[] = { 0, 0, 0, 1.0 };
	GLfloat direction2[] = { 0.0, 1.0, 0.0 };

	// Parameter fuer das Licht
	// hoehe = 30 hoehe der kugel - 4 radius - 1 puffer
	GLfloat position[] = { 0, 25, 0, 1.0 };
	GLfloat direction[] = { 0.0, -1.0, 0.0 };
	GLfloat ambient[] = { 0.5, 0.5, 0.5, 1.0 };
	GLfloat diffuse[] = { 0.8, 0.8, 0.8, 1.0 };

	// Parameter zum Zuruecksetzen
	GLfloat reset_emission[] = { 0,0,0,0 };
	GLfloat reset_ambient[] = { 0,0,0,0 };
	GLfloat reset_diffuse[] = { 0,0,0,0 };
	GLfloat reset_shininess = 0;

	// Parameter fuer die das Silbermaterial
	GLfloat silver_emission[] = { 1, 1, 1, 1.0 };
	GLfloat silver_ambient[] = { 0.23, 0.23, 0.23, 1.0 };
	GLfloat silver_diffuse[] = { 0.27, 0.27, 0.27, 1.0 };
	GLfloat silver_specular[] = { 0.77, 0.77, 0.77, 1.0 };
	GLfloat silver_shininess = 89.6;

	// Texturierung
	glBindTexture(GL_TEXTURE_2D , environment);
	glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_SPHERE_MAP);
	glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, GL_SPHERE_MAP);

	glEnable(GL_TEXTURE_2D);
	glEnable(GL_TEXTURE_GEN_S);
	glEnable(GL_TEXTURE_GEN_T);
	
	// Kugel (die Lampe ueber dem Aquarium)
	glPushMatrix();
		glTranslatef(0, 30, 0);
		glMaterialfv(GL_FRONT, GL_EMISSION, silver_emission);
		glMaterialfv(GL_FRONT, GL_AMBIENT, silver_ambient);
		glMaterialfv(GL_FRONT, GL_DIFFUSE, silver_diffuse);
		glMaterialfv(GL_FRONT, GL_SPECULAR, silver_specular);
		glMaterialf(GL_FRONT, GL_SHININESS, silver_shininess);
		glutSolidSphere(8,10,10);
	glPopMatrix();

	glDisable(GL_TEXTURE_GEN_S);
	glDisable(GL_TEXTURE_GEN_T);
	glDisable(GL_TEXTURE_2D);

	// Die Kugel leuchtet nach unten
	glLightfv(GL_LIGHT1, GL_SPOT_DIRECTION, direction);
	glLightfv(GL_LIGHT1, GL_POSITION, position);
	glLightfv(GL_LIGHT1, GL_AMBIENT, ambient);
	glLightfv(GL_LIGHT1, GL_DIFFUSE, diffuse);

	// Zweites Licht, das die Kugel anleuchtet
	glLightfv(GL_LIGHT5, GL_SPOT_DIRECTION, direction2);
	glLightfv(GL_LIGHT5, GL_POSITION, position2);
	glLightfv(GL_LIGHT5, GL_AMBIENT, ambient);
	glLightfv(GL_LIGHT5, GL_DIFFUSE, diffuse);

	glLightf(GL_LIGHT1, GL_SPOT_CUTOFF, 40.0); // kleiner Winkel, damit man den Sport erkennt
	glLightf(GL_LIGHT1, GL_SPOT_EXPONENT, 20.0);
	
	glLightf(GL_LIGHT5, GL_SPOT_CUTOFF, 30.0); // kleiner Winkel, damit man den Sport erkennt
	glLightf(GL_LIGHT5, GL_SPOT_EXPONENT, 5.0);

	glEnable(GL_LIGHT1);
	glEnable(GL_LIGHT5);
	glEnable(GL_LIGHTING);

	// Materialien wieder zuruecksetzen
	glMaterialfv(GL_FRONT, GL_EMISSION, reset_emission);
	glMaterialfv(GL_FRONT, GL_AMBIENT, reset_ambient);
	glMaterialfv(GL_FRONT, GL_DIFFUSE, reset_diffuse);
	glMaterialf(GL_FRONT, GL_SHININESS, reset_shininess);

	glEnable(GL_COLOR_MATERIAL);
}

/**
 * Zeichnet den Boden innen (kann beleuchtet werden)
 * Quellen: http://www.opengl.org/resources/code/samples/mjktips/grid/index.html
 */
void drawInnenboden(void) 
{
	// Elevation Grid (mehrere Flaechen ergeben eine bessere Beleuchtung)
	GLfloat grid2x2[2][2][3] = 
	{
		{
			{-15, 0.0, -30},
			{15, 0.0, -30}
		},
		{
			{-15, 0.0, 30},
			{15, 0.0, 30}
		}
	};

	GLfloat *grid = &grid2x2[0][0][0];
	int uSize=2;
	int vSize=2;
	int gridSize=60;

	glPushMatrix();
		// evaluator auf dem boden
		glEnable(GL_MAP2_VERTEX_3);
		glTranslatef(0,-14.99,0);
		glColor3f(0.6, 0.3, 0);
		glMapGrid2f(gridSize, 0.0, 1.0, gridSize, 0.0, 1.0);
		glMap2f(GL_MAP2_VERTEX_3, 0, 1, 3, uSize, 0,1, uSize*3, vSize, grid);
		glEvalMesh2(GL_FILL, 0, gridSize, 0, gridSize);
	glPopMatrix();
}


/**
 * Zeichnet das Aquarium
 */
void drawAquarium(void) 
{	
	//Transparenz aktivieren
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	// Glaskaefig
	glPushMatrix();
		glScalef(15, 15, 30);
		drawGlaskaefig();
	glPopMatrix();

	// Boden innen
	drawInnenboden();

	// schwarze Decke
	drawDecke();

	// Zeichnet die texturierte Pflanze
	drawPflanze();

	glDisable(GL_BLEND);

	// Wasser
	//drawWasser(); //TODO
}

/**
 * Zeichnet den Tisch
 */
void drawTisch(void) 
{
	// Tischplatte
	drawTischplatte();

	// Tischbein: vogelperspektive rechts unten
	drawTischbein(17, -16, 26);
	// Tischbein: vogelperspektive rechts oben
	drawTischbein(17, -16, -26);
	// Tischbein: vogelperspektive links unten
	drawTischbein(-17, -16, 26);
	// Tischbein: vogelperspektive links oben
	drawTischbein(-17, -16, -26);
}

/**
 * Zeichnet die Tischplatte
 */
void drawTischplatte(void) 
{
	glPushMatrix();
		glColor3f(0.6, 0.3, 0);	
		glTranslatef(0, -16.5, 0);
		glScalef(40, 3, 62);
		glutSolidCube(1);
	glPopMatrix();
}

/**
 * Zeichnet ein Tischbein an angegebener Position
 * @param x -> X-Koordinate
 * @param y -> Y-Koordinate
 * @param z -> Z-Koordinate
 */
void drawTischbein(float x, float y, float z) 
{
	int radius = 2;
	int height = 20;

	glPushMatrix();
		glColor3f(0.7, 0.5, 0);	
		glTranslatef(x, y, z);
		glRotatef(90, 1, 0, 0);
		gluCylinder(gluNewQuadric(), radius, radius, height, 20, 1);
	glPopMatrix();
}

/**
 * Zeichnet den Glaskaefig (des Aquariums)
 */
void drawGlaskaefig(void) 
{
	float tmp[3];
	glColor4f(0.3, 0.3, 0.5, 0.7);

	normalenVektor(tmp,  -1, 1, 1,  -1, -1, 1, 1, -1, 1); glNormal3fv(tmp);
	//0,0,1
	glBegin(GL_TRIANGLE_FAN);
		// vorderes Quadrat
		glVertex3f(-1, 1, 1); //0 links oben vorn
		glVertex3f(-1, -1, 1); //1 links unten vorn
		glVertex3f(1, -1, 1); //2 rechts unten vorn
		glVertex3f(1, 1, 1); //3 rechts oben vorn
	glEnd();

	normalenVektor(tmp,  -1, -1, -1, -1, 1, -1, 1, -1, -1); glNormal3fv(tmp);
	//0,0,-1
	glBegin(GL_TRIANGLE_FAN);
		// hinteres Quadrat
		glVertex3f(-1, 1, -1); //4 links oben hinten
		glVertex3f(-1, -1, -1); //5 links unten hinten
		glVertex3f(1, -1, -1); //6 rechts oben hinten
		glVertex3f(1, 1, -1); //7 rechts unten hinten
	glEnd();

	glColor4f(0.3, 0.3, 0.5, 0.3);
	normalenVektor(tmp,  1, 1, 1, 1, -1, 1, 1, -1, -1); glNormal3fv(tmp);
	//1,0,0
	glBegin(GL_TRIANGLE_FAN);
		// rechtes Quadrat
		glVertex3f(1, 1, 1); //3 rechts oben vorn
		glVertex3f(1, -1, 1); //2 rechts unten vorn
		glVertex3f(1, -1, -1); //6 rechts oben hinten
		glVertex3f(1, 1, -1); //7 rechts unten hinten		
	glEnd();

	normalenVektor(tmp,  -1, 1, 1,  -1, 1, -1, -1, -1, -1); glNormal3fv(tmp);
	//-1,0,0
	glBegin(GL_TRIANGLE_FAN);
		// linkes Quadrat
		glVertex3f(-1, 1, 1); //0 links oben vorn
		glVertex3f(-1, 1, -1); //4 links oben hinten
		glVertex3f(-1, -1, -1); //5 links unten hinten
		glVertex3f(-1, -1, 1); //1 links unten vorn		
	glEnd();
}

/**
 * Zeichnet das Wasser (im Aquariums)
 */
void drawWasser(void) 
{
	glPushMatrix();
		glColor4f(0.7, 0.7, 1, 0.5);
		glTranslatef(0, -1, 0);
		glScalef(30, 27, 60);
		glutSolidCube(1);
	glPopMatrix();
}

/**
 * Zeichnet die Decke (des Aquariums)
 */
void drawDecke(void) 
{
	glPushMatrix();
		glColor4f(0, 0, 0, 0.4);
		glTranslatef(0, 15, 0);
		glScalef(30, 1, 60);
		glutSolidCube(1);
	glPopMatrix();
}

/**
 * Drehe die Kamera link/rechts
 * Quellen: http://www.lighthouse3d.com/opengl/glut/index.php?8
 */
void orientMe(float ang) 
{
	lx = sin(ang);
	lz = -cos(ang);
	glLoadIdentity();
	gluLookAt(x, y, z, x+lx, y+ly, z+lz, 0.0f, 1.0f, 0.0f);
}

/**
 * Bewege die kamera vorne/hinten
 */
void moveMeFlat(float i) 
{
	x = x + i*(lx)*0.1;
	z = z + i*(lz)*0.1;
	glLoadIdentity();
	gluLookAt(x, y, z, x+lx, y+ly, z+lz, 0.0f, 1.0f, 0.0f);
}


/**
 * Loads the specified bitmap file from disk and copies it into an OpenGL texture.
 * Returns the GLuint representing the texture (calls exit(1) if the bitmap fails to load).
 * Quellen: http://chaoslizard.sourceforge.net/glbmp/tutorial
 */
GLuint LoadTexture(const char * bitmap_file)
{
   GLuint texture = 0; //OpenGL texture to create and return
   glbmp_t bitmap;     //object to fill with data from glbmp
 
   //try to load the specified file--if it fails, dip out
   if(!glbmp_LoadBitmap(bitmap_file, 0, &bitmap))
   {
      fprintf(stderr, "Error loading bitmap file: %s\n", bitmap_file);
   }

   //generate and bind the OpenGL texture
   glGenTextures(1, &texture);
   glBindTexture(GL_TEXTURE_2D, texture);

   //copy data from bitmap into mipmap
   if(gluBuild2DMipmaps(GL_TEXTURE_2D, 3, bitmap.width, bitmap.height, GL_RGB, GL_UNSIGNED_BYTE, bitmap.rgb_data) != 0)
   {
	   printf("Error generating bitmap file: %s\n" , bitmap_file);
   }

   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);	   
 
   //set up texture filtering
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

   glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE );
 
   //free the bitmap
   glbmp_FreeBitmap(&bitmap);
 
   return texture;
}

/**
 * die Funktion display enthaelt alle Befehle, die zum Zeichnen des
 * Bildes benoetigt werden
 */
void display(void)
{
	// Bewegung vorne/hinten
	if (deltaMove) {
		moveMeFlat(deltaMove);
	}

	// Drehung links/rechts
	if (deltaAngle) {
		angle += deltaAngle;
		orientMe(angle);
		if(updater) {
			deltaAngle=0;
			updater = false;
		}
	}
	
	// zunaechst wird das bisher gezeichnete Bild geloescht,
	// indem das Bild komplett in der Hintergrundfarbe gezeichnet
	// wird. Ausserdem wird die Verdeckungsrechnung initialisiert
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	// Raubfische zuruecksetzen
	for(int i=0; i<anzahlRaubfische; i++) 
	{
		raubfische[i]->resetNachbar();
	}

	// Fische untereinander bekannnt machen
	// Fuer Schwarmverhalten und Kollisionsvermeidung
	for(int i=0; i<anzahlFische; i++) 
	{
		fische[i]->resetNachbar();
		for(int j=0; j<anzahlFische; j++) 
		{
			if(i != j) 
			{
				fische[i]->checkNachbar(fische[j]);
			}
		}

		// Sind Raubfische in der Naehe?
		for(int j=0; j<anzahlRaubfische; j++) 
		{
			fische[i]->checkRaubfisch(raubfische[j]);
		}
	}

	
	for(int i=0; i<anzahlFische; i++) 
	{
		fische[i]->calculate(); // Normale Bewegung berechnen und Richtung bei Wandkollision aendern
		fische[i]->display(); // Fischinstanzen zeichnen
	}

	// Raubfische untereinander bekannnt machen
	// Fuer Kollisionsvermeidung
	for(int i=0; i<anzahlRaubfische; i++) 
	{
		for(int j=0; j<anzahlRaubfische; j++) 
		{
			if(i != j) 
			{
				raubfische[i]->checkNachbar(raubfische[j]);
			}
		}
	}

	// Normale Bewegung berechnen und Richtung bei Wandkollision aendern
	for(int i=0; i<anzahlRaubfische; i++) 
	{
		raubfische[i]->calculate();
		raubfische[i]->display();
	}

	// Textur laden
	environment = environment == NULL ? LoadTexture("env.bmp") : environment;
	pflanze = pflanze == NULL ? LoadTexture("plant2.bmp") : pflanze;

	// Zeichne Aquarium
	drawAquarium();

	// Zeichne Tisch
	drawTisch();

	// Zeichne Atmosphaere (Nebel, Hintergrund, ...)
	drawAthmosphere();

	// Zeichne das Cockpit
	drawCockpit();

	// Sorge dafuer, dass wirklich alle glBefehle abgearbeitet und
	// nicht gepuffert werden
	glFlush();

	// Vertausche die Backbuffer und Frontbuffer: das fertig
	// gezeichnete Bild soll jetzt angezeigt werden
	glutSwapBuffers();

	// Sorge dafuer, dass das Bild im naechsten Schleifendurchlauf
	// der glutMainLoop wieder gezeichnet wird
	glutPostRedisplay();

}


/**
 * die Funktion reshape enthaelt alle Befehle, die man benoetigt,
 * um auf Veraenderungen der Fenstergroee zu reagieren
 * @param w -> Breite des Fensters
 * @param h -> Hoehe des Fensters
 */
void reshape(int w, int h)
{
	// Das neue Seitenverhaeltnis merken wir uns
	ratio = 1.0f * w / h;

	// die folgenden drei Befehle dienen dazu anzugeben, von wo aus
	// die Szene betrachtet wird (d.h. unter welchem Blickwinkel
	// das Bild berechnet wird
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	
	// Der Viewport soll das ganze Fenster sein
    glViewport(0, 0, w, h);

	// Set the clipping volume
	gluPerspective(45, ratio, 1, 1000);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	gluLookAt(x, y, z, 
		      x + lx,y + ly,z + lz,
			  0.0f,1.0f,0.0f);

}

/*
 * die Funktion wird aufgerufen, sobald eine Taste der Tastur
 * gedrueckt wurde. Es steht die Info zur Verfuegung, welche
 * Taste gedrueckt wurde und wo sich die Maus zu diesem Zeitpunkt
 * befand
 * @param key -> gedrueckte Taste (char)
 * @param x -> Mausposition X-Koordinate
 * @param y -> Mausposition Y-Koordinate
 */
void keyboard(unsigned char key, int x, int y) 
{
	switch (key) {
		case '+': 
			// Helligkeit eines Fisches hoch
			fische[0]->powerUp();
			break;
		case '-': 
			// Helligkeit eines Fisch runter
			fische[0]->powerDown();
			break;
		case 'p':
			// Wo bin ich?
			printf("\n\nx,y,z: %f,%f,%f",x,y,z);printf("\nlx,ly,lz: %f,%f,%f",lx,ly,lz);
			break;
		case '1':
			// Alle Raubfische bleiben stehen
			for(int j=0; j<anzahlRaubfische; j++) 
			{
				raubfische[j]->bewegungStop();
			}
			
			// Alle Fische bleiben stehen
			for(int i=0; i<anzahlFische; i++) 
			{
				fische[i]->bewegungStop();
			}
			break;
		case '2':
			printf("well done\n");
			break;
		case '3':
			// Alle Raubfische schwimmen an die Oberflche
			for(int j=0; j<anzahlRaubfische; j++) 
			{
				raubfische[j]->moveUp();
			}
			
			// Alle Fische schwimmen an die Oberflche
			for(int i=0; i<anzahlFische; i++) 
			{
				fische[i]->moveUp();
			}
			break;
		case '4':
			// Alle Raubfische machen sich tot
			for(int j=0; j<anzahlRaubfische; j++) 
			{
				raubfische[j]->die();
			}
			
			// Alle Fische machen sich tot
			for(int i=0; i<anzahlFische; i++) 
			{
				fische[i]->die();
			}
			break;
	}
}

/**
 * die Funktion wird aufgerufen, sobald eine Taste der Tastur
 * gedrueckt wurde. Es steht die Info zur Verfuegung, welche
 * Taste gedrueckt wurde und wo sich die Maus zu diesem Zeitpunkt
 * befand
 * @param key -> gedrueckte Taste (int)
 * @param x -> Mausposition X-Koordinate
 * @param y -> Mausposition Y-Koordinate
 */
void pressKey(int key, int x, int y) 
{
	switch (key) {
		case GLUT_KEY_LEFT : 
			deltaAngle = CAM_ROTATE * -1;
			break;
		case GLUT_KEY_RIGHT : 
			deltaAngle = CAM_ROTATE;
			break;
		case GLUT_KEY_UP : 
			deltaMove = CAM_SPEED;
			break;
		case GLUT_KEY_DOWN : 
			deltaMove = CAM_SPEED * -1;
			break;
	}
}

/**
 * die Funktion wird aufgerufen, sobald eine Taste der Tastur
 * gedrueckt wurde. Es steht die Info zur Verfuegung, welche
 * Taste gedrueckt wurde und wo sich die Maus zu diesem Zeitpunkt
 * befand
 * @param key -> gedrueckte Taste (int)
 * @param x -> Mausposition X-Koordinate
 * @param y -> Mausposition Y-Koordinate
 */
void releaseKey(int key, int x, int y) 
{
	switch (key) {
		case GLUT_KEY_LEFT : 
		case GLUT_KEY_RIGHT : 
			deltaAngle = 0.0f;
			break;
		case GLUT_KEY_UP : 
		case GLUT_KEY_DOWN : 
			deltaMove = 0;
			break;
	}
}

/**
 * die Funktion wird aufgerufen, sobald eine Taste der Maus
 * gedrueckt wurde. Es steht die Info zur Verfuegung, welche
 * Taste betroffen ist, ob sie gedrueckt oder losgelassen wurde
 * und wo sich die Maus zu diesem Zeitpunkt befand
 * @param button -> linke/rechte/mittlere Maustaste
 * @param state -> status (DOWN/UP)
 * @param u -> x-Koordinate
 * @param v -> y-Koordinate
 */
void meineMaus(int button, int state, int u, int v) 
{
	if(GLUT_LEFT == button && GLUT_DOWN == state) 
	{
		viewnummer = viewnummer++ % 3;

		switch(viewnummer) {
			case 1: // Vogelperspektive
				x = 0;
				y = 23;
				z = 0;
				lx = 0;
				ly = -180;
				lz = 0;
				angle = 1.5707963260;
				deltaAngle = 0.0000000008; // Workaround
				updater = true;
			break;
			case 2: // Kamera innen
				x = -12.962127;
				y = 1.75;
				z = -27.929771;
				lx = 0.501494;
				ly = 0;
				lz = 0.865161;
				updater = true;
			break;
			case 3: // Kamera aussen
				x = -77.746742;
				y = 1.75;
				z = -47.038883;
				lx = 0.870534;
				ly = 0;
				lz = 0.492109;
				updater = true;
			break;
		}

		glLoadIdentity();
		gluLookAt(x, y, z, x+lx, y+ly, z+lz, 0.0f, 1.0f, 0.0f);
	}
}

/**
 * Applikation
 * @param argc -> Anzahl der Argumente (keins wird gebraucht)
 * @param argv -> Argumente
 */
int main(int argc, char** argv)
{
	char* namen[] = {"lara","leon","lena","lukas","mia","tim","laura","max","nele","jonas","lisa","paul","alina","jan"};
	const int anzahlNamen = 14;

	// Random neu initialisieren
	srand((unsigned int)time((time_t *)NULL));

	// Fischinstanzen erzeugen
	for(int i=0; i<anzahlFische; i++) 
	{	
		float x = (rand()%14);
		float y = (rand()%14);
		float z = (rand()%14);
		
		x *= rand()%2 == 0 ? -1 : 1;
		y *= rand()%2 == 0 ? -1 : 1;
		z *= rand()%2 == 0 ? -1 : 1;

		Fisch* tmp = new Fisch(namen[i%anzahlNamen], x, y, z);
		fische[i] = tmp;
	}

	// Raubfischinstanzen erzeugen
	for(int i=0; i<anzahlRaubfische; i++) 
	{
		Fisch* tmp = new Fisch("EVIL", i, i, i);
		tmp->setEvil(true);
		raubfische[i] = tmp;
	}

    // Initialisiere die GLUT
	glutInit(&argc, argv);

	// Setze den Display Mode: Farben werden in RGB + Alpha angegeben,
	// es wird eine Verdeckungsrechnung durchgefuehrt und mit
	// zwei Buffern gearbeitet
    glutInitDisplayMode (GLUT_RGBA | GLUT_DEPTH | GLUT_DOUBLE);
	
	// Setze die Fenstergroee auf 640 x 360
    glutInitWindowSize(640,480); 

	// Erzeuge ein Fenster auf dem Bildschirm an Position
	// 100, 100 (Bildschirmkoordinaten)
    glutInitWindowPosition(100,100); 

	// Erzeuge das Fenster
    glutCreateWindow("OpenGL Aquarium");

	// Bestimme, welcher Callback zum Zeichnen aufgerufen wird
	glutDisplayFunc(display);

	// Bestimme, welcher Callback bei Betaetigen einer Maustaste
	// aufgerufen wird
	glutMouseFunc(meineMaus);

	// fuer eine gedruecktgehaltene Taste wollen wir keine tausend Zeichen
	// (dafr arbeiten wir schlielich mit onkeyup/onkeydown Triggern)
	glutIgnoreKeyRepeat(1);

	// Bestimme, welcher Callback bei Betaetigung der Tastatur
	// aufgerufen wird
	glutKeyboardFunc(keyboard);
	glutSpecialFunc(pressKey);
	glutSpecialUpFunc(releaseKey);

	// Bestimme, welcher Callback bei Groeenaenderung des Fensters
	// aufgerufen wird
	glutReshapeFunc(reshape);

	// Bestimme, welcher Callback bei Untaetigkeit der GLUT
	// aufgerufen wird
	glutIdleFunc(NULL);

	// die Hintergrundfarbe wird auf blau gesetzt
	glClearColor(0.5, 0.5, 0.7, 1.0);

	// die Initialisierung der Verdeckungsrechnung ist 1.0
	glClearDepth(1.0);

	// die Verdeckungsrechnung wird eingeschaltet
	glEnable(GL_DEPTH_TEST);

	// Die vorhandenen Objektfarben kann man als Materialeigenschaften uebernehmen
	glEnable(GL_COLOR_MATERIAL);

	// uebergebe die Kontrolle nun an GLUT, die in einer Endlos-
	// schleife die Callbacks aufruft
    glutMainLoop();
	return 0;
}