/*
 * Copyright 2007 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.google.gwt.sample.mail.client;

import java.util.ArrayList;
import java.util.HashMap;

import com.google.gwt.core.client.*;
import com.google.gwt.user.client.*;
import com.google.gwt.user.client.ui.*;

/**
 * This application demonstrates how to construct a relatively complex user
 * interface, similar to many common email readers. It has no back-end,
 * populating its components with hard-coded data.
 */
public class Mail implements EntryPoint, HistoryListener, WindowResizeListener {

	private static Mail singleton;

	/**
	 * Instantiate an application-level image bundle. This object will provide
	 * programmatic access to all the images needed by widgets.
	 */
	private static final Images images = (Images) GWT.create(Images.class);

	/**
	 * An aggragate image bundle that pulls together all the images for this
	 * application into a single bundle.
	 */
	public interface Images extends Shortcuts.Images, TopPanel.Images {
	}

	/**
	 * Gets the singleton Mail instance.
	 */
	public static Mail get() {
		return singleton;
	}

	private TopPanel topPanel = new TopPanel(images);
	private VerticalPanel rightPanel = new VerticalPanel();
	private Shortcuts list = new Shortcuts(images);
	private DockPanel outer;
	private Content contents;
	private HashMap historyList = new HashMap();

	/**
	 * This method constructs the application user interface by instantiating
	 * controls and hooking up event listeners.
	 */
	public void onModuleLoad() {
		singleton = this;
		topPanel.setWidth("100%");
		
		// Create a dock panel that will contain the menu bar at the top,
		// the shortcuts to the left, and the mail list & details taking the rest.
		outer = new DockPanel();
		outer.add(topPanel, DockPanel.NORTH);
		outer.add(list, DockPanel.WEST);
		outer.add(rightPanel, DockPanel.CENTER);
		outer.setWidth("100%");

		outer.setSpacing(4);
		outer.setCellWidth(rightPanel, "100%");
		
		// Main Page
		rightPanel(new Content("Home", new Home()));
		
		// Hook the history event
		History.addHistoryListener(this);

		// Hook the window resize event, so that we can adjust the UI.
		Window.addWindowResizeListener(this);

		// Get rid of scrollbars, and clear out the window's built-in margin,
		// because we want to take advantage of the entire client area.
		Window.enableScrolling(false);
		Window.setMargin("0px");

		// Finally, add the outer panel to the RootPanel, so that it will be
		// displayed.
		RootPanel.get().add(outer);
		
		// Call the window resized handler to get the initial sizes setup. Doing
		// this in a deferred command causes it to occur after all widgets'
		// sizes
		// have been computed by the browser.
		DeferredCommand.addCommand(new Command() {
			public void execute() {
				onWindowResized(Window.getClientWidth(), Window
						.getClientHeight());
			}
		});

		onWindowResized(Window.getClientWidth(), Window.getClientHeight());
	}

	/**
	 * adds a list of widgets to the main panel
	 * @param content
	 */
	public void rightPanel(Content content) {
		if (contents == content) {
			return;
		} else {
			contents = content;
		}
		
		rightPanel.clear();
		for (int i = 0; i < contents.size(); i++) {
			rightPanel.add(contents.get(i));
		}
	}

	/**
	 * adds a list of widgets to the top panel
	 * @param al
	 */
	public void topPanel(ArrayList al) {
		topPanel.clearMenu();
		for (int i = 0; i < al.size(); i++) {
			topPanel.addMenuItem((Widget) al.get(i));
		}
	}

	/**
	 * 
	 * @param width
	 * @param height
	 */
	public void onWindowResized(int width, int height) {
		// Adjust the shortcut panel and detail area to take up the available
		// room in the window.
		int shortcutHeight = height - list.getAbsoluteTop() - 8;
		if (shortcutHeight < 1) {
			shortcutHeight = 1;
		}
		list.setHeight("" + shortcutHeight);

		// Give the mail detail widget a chance to resize itself as well.
		for (int i = 0; i < contents.size(); i++) {
			((Composite) contents.get(i)).setPixelSize(width, height);
		}
	}
	
	/**
	 * adds item to browser history
	 * @param sink
	 */
	public void addHistory(Sink sink) {
		History.newItem(sink.getName());
		historyList.put(sink.getName(), sink);
	}
	
	/**
	 * adds item to browser history
	 * @param path
	 * @param sink
	 */
	public void addHistory(String path, Sink sink) {
		History.newItem(path);
		historyList.put(path, sink);
	}
	
	/**
	 * 
	 * @param historyToken
	 */
	public void onHistoryChanged(String historyToken) {
		historyToken = historyToken.equals("") ? "Home" : historyToken;
		// Find the SinkInfo associated with the history context. If one is
	    // found, show it (It may not be found, for example, when the user mis-
	    // types a URL, or on startup, when the first context will be "").
	    Sink info = (Sink) historyList.get(historyToken);
	    if (info == null) {
	      ErrorDialog dlg = new ErrorDialog();
	      dlg.show();
	      dlg.center();
	      return;
	    }
	    
	    // change menu style
	    list.setShortcut(info.getShortcut() < 0 ? 0 : info.getShortcut());
	    
	    // show content
	    if(historyToken.indexOf("/") > 0) {
	    	String param = historyToken.substring(historyToken.indexOf("/"));
	    	info.show(param);
	    } else {
	    	info.show();
	    }
	}
}
