/*
 * Copyright 2007 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.google.gwt.sample.mail.client;

import com.google.gwt.sample.mail.client.ui.calendar.Calendar;
import com.google.gwt.sample.mail.client.ui.contacts.Contacts;
import com.google.gwt.sample.mail.client.ui.mail.Mailboxes;
import com.google.gwt.sample.mail.client.ui.notices.Notices;
import com.google.gwt.sample.mail.client.ui.tasks.Tasks;
import com.google.gwt.user.client.*;
import com.google.gwt.user.client.ui.*;

/**
 * A composite that contains the shortcut stack panel on the left side. The
 * mailbox tree and shortcut lists don't actually do anything, but serve to show
 * how you can construct an interface using
 * {@link com.google.gwt.user.client.ui.StackPanel},
 * {@link com.google.gwt.user.client.ui.Tree}, and other custom widgets.
 */
public class Shortcuts extends Composite {

	/**
	 * An image bundle specifying the images for this Widget and aggragating
	 * images needed in child widgets.
	 */
	public interface Images extends Contacts.Images, Mailboxes.Images {
		AbstractImagePrototype contactsgroup();

		AbstractImagePrototype leftCorner();

		AbstractImagePrototype mailgroup();

		AbstractImagePrototype rightCorner();

		AbstractImagePrototype tasksgroup();

		AbstractImagePrototype calendargroup();

		AbstractImagePrototype noticesgroup();
	}

	private int nextHeaderIndex = 0;

	private Sink[] shortcuts = new Sink[5];

	private StackPanel stackPanel = new StackPanel() {
		public void onBrowserEvent(Event event) {
			int oldIndex = getSelectedIndex();
			super.onBrowserEvent(event);
			int newIndex = getSelectedIndex();
			if (oldIndex != newIndex) {
				updateSelectedStyles(oldIndex, newIndex);
				show(newIndex);
			}
		}
	};

	/**
	 * Constructs a new shortcuts widget using the specified images.
	 * 
	 * @param images
	 *            a bundle that provides the images for this widget
	 */
	public Shortcuts(Shortcuts.Images images) {
		// Create the groups within the stack panel.
		shortcuts[0] = new Mailboxes(images);
		shortcuts[0].setShortcut(0);
		add(images, shortcuts[0], images.mailgroup(), "Mail");
		shortcuts[1] = new Contacts(images);
		shortcuts[1].setShortcut(1);
		add(images, shortcuts[1], images.contactsgroup(), "Kontakte");
		shortcuts[2] = new Tasks();
		shortcuts[2].setShortcut(2);
		add(images, shortcuts[2], images.tasksgroup(), "Aufgaben");
		shortcuts[3] = new Calendar(images);
		shortcuts[3].setShortcut(3);
		add(images, shortcuts[3], images.calendargroup(), "Kalender");
		shortcuts[4] = new Notices();
		shortcuts[4].setShortcut(4);
		add(images, shortcuts[4], images.noticesgroup(), "Notizen");

		initWidget(stackPanel);
	}

	/**
	 * 
	 */
	protected void onLoad() {
		// Show the mailboxes group by default.
		stackPanel.showStack(0);
		// updateSelectedStyles(-1, 0);
	}

	/**
	 * 
	 * @param images
	 * @param widget
	 * @param imageProto
	 * @param caption
	 */
	private void add(Shortcuts.Images images, Widget widget,
			AbstractImagePrototype imageProto, String caption) {
		widget.addStyleName("mail-StackContent");
		stackPanel.add(widget, createHeaderHTML(images, imageProto, caption),
				true);
	}

	/**
	 * 
	 * @param index
	 * @return
	 */
	private String computeHeaderId(int index) {
		return "header-" + this.hashCode() + "-" + index;
	}

	/**
	 * Creates an HTML fragment that places an image & caption together, for use
	 * in a group header.
	 * 
	 * @param imageProto
	 *            an image prototype for an image
	 * @param caption
	 *            the group caption
	 * @return the header HTML fragment
	 */
	private String createHeaderHTML(Shortcuts.Images images,
			AbstractImagePrototype imageProto, String caption) {

		boolean isTop = (nextHeaderIndex == 0);
		String cssId = computeHeaderId(nextHeaderIndex);
		nextHeaderIndex++;

		String captionHTML = "<table class='caption' cellpadding='0' cellspacing='0'>"
				+ "<tr><td class='lcaption'>"
				+ imageProto.getHTML()
				+ "</td><td class='rcaption'><b style='white-space:nowrap'>"
				+ caption + "</b></td></tr></table>";

		return "<table id='" + cssId
				+ "' align='left' cellpadding='0' cellspacing='0'"
				+ (isTop ? " class='is-top'" : "") + "><tbody>"
				+ "<tr><td class='box-00'>" + images.leftCorner().getHTML()
				+ "</td>" + "<td class='box-10'>&nbsp;</td>"
				+ "<td class='box-20'>" + images.rightCorner().getHTML()
				+ "</td>" + "</tr><tr>" + "<td class='box-01'>&nbsp;</td>"
				+ "<td class='box-11'>" + captionHTML + "</td>"
				+ "<td class='box-21'>&nbsp;</td>" + "</tr></tbody></table>";
	}

	/**
	 * Example of using the DOM class to do CSS class name tricks that have
	 * become common to AJAX apps. In this case we add CSS class name for the
	 * stack panel item that is below the selected item.
	 */
	private void updateSelectedStyles(int oldIndex, int newIndex) {
		oldIndex++;
		if (oldIndex > 0 && oldIndex < stackPanel.getWidgetCount()) {
			Element elem = DOM.getElementById(computeHeaderId(oldIndex));
			DOM.setElementProperty(elem, "className", "");
		}

		newIndex++;
		if (newIndex > 0 && newIndex < stackPanel.getWidgetCount()) {
			Element elem = DOM.getElementById(computeHeaderId(newIndex));
			DOM.setElementProperty(elem, "className", "is-beneath-selected");
		}
	}

	/**
	 * sets the menu
	 * 
	 * @param newIndex
	 */
	public void setShortcut(int newIndex) {
		int oldIndex = stackPanel.getSelectedIndex();
		if (newIndex != -1 && oldIndex != newIndex) {
			updateSelectedStyles(oldIndex, newIndex);
		}
		stackPanel.showStack(newIndex);
	}

	/**
	 * 
	 * @param newIndex
	 */
	public void show(int newIndex) {
		shortcuts[newIndex].show();
	}
}
