/*
 * Copyright 2007 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.google.gwt.sample.mail.client.ui.mail;

import java.util.ArrayList;

import com.google.gwt.core.client.GWT;
import com.google.gwt.sample.mail.client.*;
import com.google.gwt.sample.mail.client.rpc.MailItem;
import com.google.gwt.sample.mail.client.rpc.MailService;
import com.google.gwt.sample.mail.client.rpc.MailServiceAsync;
import com.google.gwt.sample.mail.client.ui.mail.compose.MailCompose;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.rpc.ServiceDefTarget;
import com.google.gwt.user.client.ui.*;

/**
 * A composite that displays a list of emails that can be selected.
 */
public class MailList extends Composite implements TableListener {
	private static final int VISIBLE_EMAIL_COUNT = 10;
	private MailItem currentMailItem;
	private int selectedRow = -1;
	private int lastIndex = 0;
	private FlexTable table = new FlexTable();
	private HorizontalPanel navBar = new HorizontalPanel();
	private Mailboxes parent;
	private String path;

	/**
	 * constructor
	 * 
	 * @param parent
	 *            reference to sink object (Mailboxes)
	 * @param path
	 */
	public MailList(Mailboxes parent, String path) {
		this.path = path;
		this.parent = parent;

		// Setup the table.
		table.setCellSpacing(0);
		table.setCellPadding(0);
		table.setWidth("100%");

		// Hook up events.
		table.addTableListener(this);

		initWidget(table);
		setStyleName("mail-List");

		initTable();
		update();
	}

	/**
	 * 
	 * @return path
	 */
	public String getPath() {
		return path;
	}

	/**
	 * Select the row that was clicked (-1 to account for header row).
	 * @param sender
	 * @param row
	 * @param cell
	 */
	public void onCellClicked(SourcesTableEvents sender, int row, int cell) { 
		if (row > 0) {
			selectRow(row - 1);
		}
	}

	/**
	 * Initializes the table so that it contains enough rows for a full page of
	 * emails. Also creates the images that will be used as 'read' flags.
	 */
	private void initTable() {
		// Create the header row.
		table.setText(0, 0, "Sender");
		table.setText(0, 1, "Email");
		table.setText(0, 2, "Subject");
		table.setText(0, 3, "Date");
		table.setWidget(0, 4, navBar);
		table.getRowFormatter().setStyleName(0, "mail-ListHeader");

		// Initialize the rest of the rows.
		for (int i = 0; i < VISIBLE_EMAIL_COUNT; ++i) {
			table.setText(i + 1, 0, "");
			table.setText(i + 1, 1, "");
			table.setText(i + 1, 2, "");
			table.setText(i + 1, 3, "");
			table.getCellFormatter().setWordWrap(i + 1, 0, false);
			table.getCellFormatter().setWordWrap(i + 1, 1, false);
			table.getCellFormatter().setWordWrap(i + 1, 2, false);
		}
	}

	/**
	 * Selects the given row (relative to the current page).
	 * 
	 * @param row
	 *            the row to be selected
	 */
	private void selectRow(int row) {
		// When a row (other than the first one, which is used as a header) is
		// selected, display its associated MailItem.
		styleRow(selectedRow, false);
		styleRow(row, true);
		selectedRow = row;
		
		// AJAX
		MailServiceAsync emailService = (MailServiceAsync) GWT.create(MailService.class);
		ServiceDefTarget endpoint = (ServiceDefTarget) emailService;
		String moduleRelativeURL = GWT.getModuleBaseURL() + "rpc";
		endpoint.setServiceEntryPoint(moduleRelativeURL);

		AsyncCallback callback = new AsyncCallback() {
			public void onSuccess(Object result) {
				// TODO: caching of details
				if (result == null) {
					return;
				}
				currentMailItem = (MailItem) result;
				parent.displayItem(currentMailItem);
			}

			public void onFailure(Throwable caught) {
				Window.alert(caught.getMessage());
			}
		};

		emailService.getMail(row, callback);
	}

	/**
	 * 
	 * @param row
	 * @param selected
	 */
	private void styleRow(int row, boolean selected) {
		if (row != -1) {
			if (selected) {
				table.getRowFormatter().addStyleName(row + 1,"mail-SelectedRow");
			} else {
				table.getRowFormatter().removeStyleName(row + 1,"mail-SelectedRow");
			}
		}
	}

	/**
	 * update the maillist
	 */
	private void update() {
		// (1) Create the client proxy. Note that although you are creating the
		// service interface proper, you cast the result to the asynchronous
		// version of
		// the interface. The cast is always safe because the generated proxy
		// implements the asynchronous interface automatically.
		//
		MailServiceAsync emailService = (MailServiceAsync) GWT.create(MailService.class);

		// (2) Specify the URL at which our service implementation is running.
		// Note that the target URL must reside on the same domain and port from
		// which the host page was served.
		//
		ServiceDefTarget endpoint = (ServiceDefTarget) emailService;
		String moduleRelativeURL = GWT.getModuleBaseURL() + "rpc";
		endpoint.setServiceEntryPoint(moduleRelativeURL);
		
		// (3) Create an asynchronous callback to handle the result.
		//
		AsyncCallback callback = new AsyncCallback() {
			public void onSuccess(Object result) {
				MailItem[] mails = (MailItem[]) result;

				// Add the new mails to the table
				for (int i = 0; i < mails.length; ++i) {
					MailItem item = mails[i];

					// Add a new row to the table, then set each of its columns
					// to the email's sender and subject values.
					table.setText(lastIndex+i+1, 0, item.sender);
					table.setText(lastIndex+i+1, 1, item.email);
					table.setText(lastIndex+i+1, 2, item.subject);
					table.setText(lastIndex+i+1, 3, item.date);
				}

				lastIndex = lastIndex+mails.length;

				// Select the first row if none is selected.
				if (selectedRow == -1) {
					selectRow(0);
				}
			}

			public void onFailure(Throwable caught) {
				Window.alert(caught.getMessage());
			}
		};

		// (4) Make the call. Control flow will continue immediately and later
		// 'callback' will be invoked when the RPC completes.
		//
		emailService.getMails(path, new Integer(1), callback);
		
		System.out.println("emailService.getMails("+path+", "+lastIndex+", callback)");
	}

	/**
	 * compose a new mail
	 */
	public void compose() {
		ArrayList tmp = new ArrayList();
		tmp.add(new MailCompose());
		Mail.get().rightPanel(new Content("Neue E-Mail erstellen", tmp));
	}

	/**
	 * reply to an email
	 */
	public void reply() {
		MailCompose compose = new MailCompose();
		compose.setRecipient(currentMailItem.email);
		compose.setSubject("Re: " + currentMailItem.subject);
		compose.setText("\n\n" + currentMailItem.body.replaceAll("\n", "\n>> "));

		Mail.get().rightPanel(new Content("Neue Antwort erstellen", compose));
	}

	/**
	 * delete an email
	 */
	public void delete() {
		table.setHTML(selectedRow + 1, 0, "&nbsp;");
		table.setHTML(selectedRow + 1, 1, "&nbsp;");
		table.setHTML(selectedRow + 1, 2, "&nbsp;");
		table.setHTML(selectedRow + 1, 3, "&nbsp;");
	}

	/**
	 * select next mail
	 */
	public void next() {
		selectRow(selectedRow + 1);
	}

	/**
	 * select previous mail
	 */
	public void prev() {
		if(selectedRow > 0)
			selectRow(selectedRow - 1);
	}

	/**
	 * forward the mail to another recipient
	 */
	public void forward() {
		MailCompose compose = new MailCompose();
		compose.setSubject("Fw: " + currentMailItem.subject);
		compose.setText("\n\n"+ currentMailItem.body.replaceAll("\n", "\n>> "));

		Mail.get().rightPanel(new Content("E-Mail weiterleiten", compose));
	}
}
