/*
 * Copyright 2007 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.google.gwt.sample.mail.client.ui.mail;

import java.util.ArrayList;

import com.google.gwt.core.client.GWT;
import com.google.gwt.i18n.client.Constants;
import com.google.gwt.sample.mail.client.AboutDialog;
import com.google.gwt.sample.mail.client.Content;
import com.google.gwt.sample.mail.client.Mail;
import com.google.gwt.sample.mail.client.Sink;
import com.google.gwt.sample.mail.client.rpc.MailItem;
import com.google.gwt.user.client.*;
import com.google.gwt.user.client.ui.*;

/**
 * A tree displaying a set of email folders.
 */
public class Mailboxes extends Sink {
	private MailList mailList;
	private MailDetail mailDetail = new MailDetail();
	private Language lang;

	/**
	 * Specifies the images that will be bundled for this Composite and specify
	 * that tree's images should also be included in the same bundle.
	 */
	public interface Images extends ImageBundle, TreeImages {
		AbstractImagePrototype drafts();
		AbstractImagePrototype home();
		AbstractImagePrototype inbox();
		AbstractImagePrototype sent();
		AbstractImagePrototype templates();
		AbstractImagePrototype trash();
	}
	
	/**
	 * Static string language implementation
	 */
	public interface Language extends Constants {
		String compose();
		String sync();
		String reply();
		String replyAll();
		String forward();
		String print();
		String delete();
		String spam();
		String nospam();
		String previous();
		String next();		
	}

	private Tree tree;

	/**
	 * Constructs a new mailboxes widget with a bundle of images.
	 * 
	 * @param images
	 *            a bundle that provides the images for this widget
	 */
	public Mailboxes(Images images) {
		super("Mailbox");
		tree = new Tree(images);
		
		// Language
		lang = (Language) GWT.create(Language.class);
		
		// Demo Data: E-Mail Account 1
		TreeItem account1 = new TreeItem(new HTML("<span>" + images.home().getHTML() + "foo@example.com</span>"));
		account1.addItem(new Mailbox(this, "foo@example.com//inbox", "Inbox", images.inbox()));
		account1.addItem(new Mailbox(this, "foo@example.com//inbox//drafts", "Drafts", images.drafts()));
		account1.addItem(new Mailbox(this, "foo@example.com//inbox//templates", "Templates", images.templates()));
		account1.addItem(new Mailbox(this, "foo@example.com//inbox//sent", "Sent", images.sent()));
		account1.addItem(new Mailbox(this, "foo@example.com//inbox//trash", "Trash", images.trash()));
		tree.addItem(account1);
		account1.setState(true);

		// Demo Data: E-Mail Account 2
		TreeItem account2 = new TreeItem(new HTML("<span>" + images.home().getHTML() + "foo2@example.com</span>"));
		account2.addItem(new Mailbox(this, "foo2@example.com//inbox", "Inbox", images.inbox()));
		account2.addItem(new Mailbox(this, "foo2@example.com//inbox//drafts", "Drafts", images.drafts()));
		account2.addItem(new Mailbox(this, "foo2@example.com//inbox//templates", "Templates", images.templates()));
		account2.addItem(new Mailbox(this, "foo2@example.com//inbox//sent", "Sent", images.sent()));
		account2.addItem(new Mailbox(this, "foo2@example.com//inbox//trash", "Trash", images.trash()));
		tree.addItem(account2);
		account2.setState(true);

		// init widget
		initWidget(tree);
	}

	/**
	 * Displays the specified item.
	 * 
	 * @param item
	 */
	public void displayItem(MailItem item) {
		mailDetail.setItem(item);
	}

	/**
	 * maillist without directory?
	 */
	public void show() {
		show("//inbox");
	}

	/**
	 * shows maillist, maildetail and buttons
	 * @param path
	 */
	public void show(String path) {
		mailList = new MailList(this, path);
		mailList.setWidth("100%");

		ArrayList right = new ArrayList();
		right.add(mailList);
		right.add(mailDetail);

		Mail.get().rightPanel(new Content("Mailbox: "+path, right));
		mailList.setWidth("100%");
		mailDetail.setWidth("100%");
		
		mailDetail.adjustSize(Window.getClientWidth(), Window.getClientHeight());

		ArrayList top = new ArrayList();

		top.add(new Button(lang.compose(), new ClickListener() {
			public void onClick(Widget sender) {
				mailList.compose();
			}
		}));

		top.add(new Button(lang.sync(), new ClickListener() {
			public void onClick(Widget sender) {
				AboutDialog dlg = new AboutDialog();
				dlg.show();
				dlg.center();
			}
		}));

		top.add(new Button(lang.reply(), new ClickListener() {
			public void onClick(Widget sender) {
				mailList.reply();
			}
		}));

		top.add(new Button(lang.replyAll(), new ClickListener() {
			public void onClick(Widget sender) {
				mailList.reply();
			}
		}));

		top.add(new Button(lang.forward(), new ClickListener() {
			public void onClick(Widget sender) {
				mailList.forward();
			}
		}));

		top.add(new Button(lang.print(), new ClickListener() {
			public void onClick(Widget sender) {
				Window.alert("How high?");
			}
		}));

		top.add(new Button(lang.delete(), new ClickListener() {
			public void onClick(Widget sender) {
				mailList.delete();
			}
		}));

		top.add(new Button(lang.spam(), new ClickListener() {
			public void onClick(Widget sender) {
				Window.alert("How high?");
			}
		}));

		top.add(new Button(lang.nospam(), new ClickListener() {
			public void onClick(Widget sender) {
				Window.alert("How high?");
			}
		}));

		top.add(new Button(lang.previous(), new ClickListener() {
			public void onClick(Widget sender) {
				mailList.prev();
			}
		}));

		top.add(new Button(lang.next(), new ClickListener() {
			public void onClick(Widget sender) {
				mailList.next();
			}
		}));

		Mail.get().topPanel(top);
		super.show(name+path);
	}
}