/*
 * Copyright 2007 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.google.gwt.sample.mail.server;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import com.google.gwt.sample.mail.client.rpc.*;
import com.google.gwt.user.server.rpc.*;

/**
 * Server-side implementation of the {@link MailService} interface.
 * 
 * This class is stateless. However, it also has no backing store, meaning that
 * if the server is restarted, the data and version all reset to defaults. Since
 * the server side is implicitly authoritative, a server restart will also
 * effectively wipe out any data on clients (unless the user overrides.) In a
 * real-world implementation, this servlet would have a real backing store such
 * as a database.
 * 
 * Similarly, this class has no user authentication, where a real-world
 * implementation probably would.
 * 
 * This class implements optimistic concurrency, in that it only accepts a data
 * update if the new data is reported along with the server's current version.
 * That is, if a client is not up to date and reports an old/non-current
 * version, the server rejects the update and requires the client to fetch the
 * new version before overwriting.
 */
public class MailServiceImpl extends RemoteServiceServlet implements MailService {
	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	/**
	 * Fetches a list of all notes known to the server.
	 * 
	 * @return all notes on the server; may be null on error, but will never be
	 *         empty
	 */
	public MailItem[] getMails(String path, Integer start) {
		System.out.println("server: getMails("+path+", "+start+")");
		Map data = getDataMap();
		MailItem[] mails = new MailItem[data.size()];
		Iterator it = data.values().iterator();
		for (int i = 0; it.hasNext(); ++i) {
			mails[i] = (MailItem) it.next();
		}
		return mails;
	}

	/**
	 * Instructs the server to update its state with all the indicated notes.
	 * Many clients may call this method; the server takes the union of all
	 * unique names that it sees, from all clients. If the notes' versions are
	 * up to date, they will replace any copies on the server.
	 * 
	 * See class description for summary of optimistic concurrency.
	 * 
	 * @param mails
	 *            the list of notes to create or (attempt to) overwrite
	 */
	public void setMails(MailItem[] mails) {
		System.out.println("server: setMails("+mails+")");
		if (mails == null) {
			return;
		}
		Map data = getDataMap();
		for (int i = 0; i < mails.length; ++i) {
			if (data.containsKey(mails[i].id)) {
				MailItem nd = (MailItem) data.get(mails[i].id);
				int newVer = Integer.parseInt(mails[i].getVersion());
				int oldVer = Integer.parseInt(nd.getVersion());
				if (newVer >= oldVer) {
					// update nd object
				}
			} else { 
				data.put(mails[i].id, mails[i]);
			}
		}
	}
	
	/**
	 * 
	 * @param row
	 */
	public MailItem getMail(int row) {
		System.out.println("server: getMail("+row+")");
		return MailItems.getMailItem(row);
	}

	/**
	 * Fetches all the note records known to the server.
	 * 
	 * @return a Map containing all data known to the server
	 */
	protected Map getDataMap() {
//		Map m = (Map) getServletContext().getAttribute("com.google.gearsdemo.data");
//		if (m == null) {
//			m = new HashMap();
//			m.put("default", new MailItem("sender", "email", "subject", "body"));
//			getServletContext().setAttribute("com.google.gearsdemo.data", m);
//		}

		HashMap m = new HashMap();
		for(int i=0; i<10; i++) {
			MailItem item = MailItems.getMailItem(i);
			m.put(item.id, item);
		}
		
		return m;
	}
}